/*=============================================================================
Copyright  2010 Ametek Programmable Power. All rights reserved.

FILE NAME:  AmetekDC.h

PURPOSE:
   Instrument Driver declarations for AMETEK Programmable Power XG/XTR Series DC Power Supplies.

NOTES:
   IVI Instrument Driver for AMETEK Programmable Power XG/XTR Series Power Supply. Compatible
   with Ethernet USB, RS-232, RS-485 and GPIB Interface cards. Can be used to control
   RS-485 bus instruments via a RS-232, RS-485 or GPIB equipped unit.

   Developed using LabWindows/CVI 9.1, 2009


CHANGE HISTORY:
$Log: AmetekDC.h $
	
    Initial Release.
	Date: May 12, 2010  By: AMETEK Programmable Power
   	Revision: 1.0.0.0.
   
=============================================================================*/


#ifndef __AMETEKDC_HEADER
#define __AMETEKDC_HEADER

#include <ivi.h>
#include <ividcpwr.h>

#if defined(__cplusplus) || defined(__cplusplus__)
extern "C" {
#endif

/****************************************************************************
 *----------------- Instrument Driver Revision Information -----------------*
 ****************************************************************************/
#define AMETEKDC_MAJOR_VERSION                1     /* Instrument driver major version   */
#define AMETEKDC_MINOR_VERSION                1     /* Instrument driver minor version   */
                                                                
#define AMETEKDC_CLASS_SPEC_MAJOR_VERSION     2     /* Class specification major version */
#define AMETEKDC_CLASS_SPEC_MINOR_VERSION     0     /* Class specification minor version */
                                            
#define AMETEKDC_DRIVER_VENDOR                "AMETEK Programmable Power"
#define AMETEKDC_DRIVER_DESCRIPTION           "DC Power Supply Driver"


/****************************************************************************
 *------------------------ Driver specific constants -----------------------*
 ****************************************************************************/
// Channel string associate with the master channel.  The master channel is the
// RS-485 address associated with the XG directly connected to the interface 
// being used.
#define AMETEKDC_MASTER_CHANNEL				"A"

// The boardcast channel is the channel that will be used to send to all devices
// attached to RS-485 bus and the XG attached directly to the bus.
#define AMETEKDC_BROADCAST_CHANNEL			"0"

#define AMETEKDC_IO_SESSION_TYPE              IVI_VAL_VISA_SESSION_TYPE
                    
#define AMETEKDC_VAL_RESP_TERM_CHAR			'\r'


// Minimum delay time between sending subsequent commands to the XG.
// This delay allows a certain amount of time to get a response back from the
// interface, when using slow baud rates this might become a factor.
#define AMETEKDC_MIN_CMD_DELAY_SEC			    0.02

// This is the maximum amount of time to block waiting for VISA to report
// that the transmission has completed.  If it takes longer than 5 second to
// issue one of these commands something is seriously wrong.
#define AMETEKDC_IO_COMPLETION_TIMEOUT_MS		5000

// Default value for the data rate in bps. This will be used for serial communication
// if no rate is specified in the setup string.
#define AMETEKDC_IO_DEFAULT_DATARATE			9600

/*****************************************************************************
 *--------------------- Power Supply Property Declarations -----------------------*
 *****************************************************************************/
// XG Maximum output voltage is 110%
#define MAX_OUTPUT_VOLTAGE				1.05

// XG Maximum output current is 105%
#define MAX_OUTPUT_CURRENT				1.05

// XG Default ID String
#define AMETEKDC_DEFAULT_ID_STRING	"AMETEK,XG 6-110,SN# E012345678,1.03 Build 17,21/11/2006"

// XG Default Model String (should be the same value shown in the ID string.
#define AMETEKDC_DEFAULT_MODEL_STRING		"XG 6-110"

// XG Default Definition ID String. This is used for setting a new ID string. 
#define AMETEKDC_DEFAULT_DEF_ID_STRING "AMETEK,%s,SN# E00000000,1.03 Build 17,21/11/2006"

// XG minimum foldback delay time.
#define AMETEKDC_FOLDBACK_MIN_DELAY_SEC	0.5

// Maximum amount of time to delay between autosequence steps.
#define AMETEKDC_MAX_DWELL_TIME_SEC		180

// Default address of GPIB and the MCHANNEL
#define AMETEKDC_DEFAULT_ADDRESS			2

// The number of user setting slots.
#define AMETEKDC_NUMB_USER_SETTINGS			3

/*----- I/O buffer sizes -----------------------------------------------------*/
#define BUFFER_SIZE              1024L//512L
#define SMALL_BUFF_SIZE          64L

/****************************************************************************
 *------------------------ Error And Completion Codes ----------------------*
 ****************************************************************************/
#define AMETEKDC_ERROR_TRIGGER_NOT_SOFTWARE       IVIDCPWR_ERROR_TRIGGER_NOT_SOFTWARE
#define AMETEKDC_ERROR_INVALID_PARAMETER          IVI_ERROR_INVALID_PARAMETER

/****************************************************************************
 *--------- Obsolete Inherent Instrument Attributes and functions ----------*
 ****************************************************************************/
/* These attributes and functions have been deprecated and may not
   be supported in future versions of this driver. */
 
/*------ Driver Information ------------------------------------------------*/
#define AMETEKDC_ATTR_SPECIFIC_DRIVER_MAJOR_VERSION  IVI_ATTR_SPECIFIC_DRIVER_MAJOR_VERSION  /* ViInt32,  read-only  */
#define AMETEKDC_ATTR_SPECIFIC_DRIVER_MINOR_VERSION  IVI_ATTR_SPECIFIC_DRIVER_MINOR_VERSION  /* ViInt32,  read-only  */

/*------ Error Info --------------------------------------------------------*/
#define AMETEKDC_ATTR_PRIMARY_ERROR                  IVI_ATTR_PRIMARY_ERROR              /* ViInt32   */
#define AMETEKDC_ATTR_SECONDARY_ERROR                IVI_ATTR_SECONDARY_ERROR            /* ViInt32   */
#define AMETEKDC_ATTR_ERROR_ELABORATION              IVI_ATTR_ERROR_ELABORATION          /* ViString  */

/*------ Advanced Session Information --------------------------------------*/
#define AMETEKDC_ATTR_IO_SESSION_TYPE                IVI_ATTR_IO_SESSION_TYPE            /* ViString, read-only  */
#define AMETEKDC_ATTR_IO_SESSION                     IVI_ATTR_IO_SESSION                 /* ViSession, read-only */

/*------ Deprecated Attribute Identifiers for Renamed Attributes -----------*/
#define AMETEKDC_ATTR_NUM_CHANNELS                   AMETEKDC_ATTR_CHANNEL_COUNT           /* ViInt32,  read-only  */
#define AMETEKDC_ATTR_QUERY_INSTR_STATUS             AMETEKDC_ATTR_QUERY_INSTRUMENT_STATUS /* ViBoolean */
#define AMETEKDC_ATTR_RESOURCE_DESCRIPTOR            AMETEKDC_ATTR_IO_RESOURCE_DESCRIPTOR  /* ViString, read-only  */



// Bit positions for the various protection registers.  This includes the alarm mask register
// and output latch registers. They are also XG specific.
typedef enum PRT_eFLAG
{
	PRT_eFLAG_ACFAIL = 0,
	PRT_eFLAG_OTP,
	PRT_eFLAG_OTPSLAVE,
	PRT_eFLAG_FAN,
	PRT_eFLAG_FANSLAVE,
	PRT_eFLAG_INTLOCK,
	PRT_eFLAG_OVP,
	PRT_eFLAG_OCP,
	PRT_eFLAG_FLD,
	PRT_eFLAG_SD,
	PRT_eFLAG_NONE
}PRT_teFLAG;


/**************************************************************************** 
 *-------------------------- 488.2 Status Register Bit Positions -----------------------------* 
 ****************************************************************************/
// Bit flag defintions for the SCPI Operation Status Register.
typedef enum eSBYTE
{
	eSBYTE_ERR = 2,
	eSBYTE_QSR,
	eSBYTE_MAV,
	eSBYTE_SESR,
	eSBYTE_RQS,
	eSBYTE_OSR
}teSBYTE;

// Bit flag defintions for the SCPI Operation Status Register.
typedef enum eSESR
{
	eSESR_OPC = 0,
	eSESR_QYE = 2 ,
	eSESR_DDE,
	eSESR_EXE,
	eSESR_CME
}teSESR;

/**************************************************************************** 
 *-------------------------- SCPI Status Register Bit Positions -----------------------------* 
 ****************************************************************************/

// Bit flag defintions for the SCPI Operation Status Register.
typedef enum eOPER_STATUS
{
	eOPER_STATUS_CAL,
	eOPER_STATUS_MEAS = 4,
	eOPER_STATUS_SD = 9,
	eOPER_STATUS_LLOC = 10,
	eOPER_STATUS_CSHARE = 11,
	eOPER_STATUS_CV = 12,
	eOPER_STATUS_CC = 13,
	eOPER_STATUS_PROG_RUN = 14
}teOPER_STATUS;

// Bit positions for the Current Sharing fan-out register attached 
// to the operations register.
typedef enum eOPER_CSH_SREG
{
	eOPER_CSH_SREG_MASTER,
	eOPER_CSH_SREG_SLAVE	
}teOPER_CSH_SREG;

// Bit positions for the Shutdown fan-out register attached 
// to the operations register.
typedef enum eOPER_SD_SREG
{
	eOPER_SD_STATUS_PROT,
	eOPER_SD_STATUS_INT,
	eOPER_SD_STATUS_EXTSD = 3	
}teOPER_SD_STATUS;

// Bit positions for the protection fan-out register attached
// to the shutdown status register and subsequent operations
// status register.
typedef enum eOPER_SD_PROT_SREG
{
	eOPER_SD_PROT_SREG_OVP,
	eOPER_SD_PROT_SREG_UVP,
	eOPER_SD_PROT_SREG_OCP,		
	eOPER_SD_PROT_SREG_ACOFF = 3,
	eOPER_SD_PROT_SREG_OTP = 4,
	eOPER_SD_PROT_SREG_FLD = 7,
	eOPER_SD_PROT_SREG_FAN = 8
}teOPER_SD_PROT_SREG;

// Bit positions for the Questionable Status register.
typedef enum eQUES_STATUS
{
	eQUES_STATUS_VOLT,
	eQUES_STATUS_TEMP = 4,
	eQUES_STATUS_CAL = 8,
	eQUES_STATUS_AC_OFF = 11,
}teQUES_STATUS;

// Bit positions for the Questionable Voltage Status register.
typedef enum eQUES_VOLT_STATUS
{
	eQUES_VOLT_STATUS_OVV,			// Over voltage protection
	eQUES_VOLT_STATUS_UDV			// Under voltage protection
}teQUES_VOLT_STATUS;

// Bit positions for the Questionable Temperature Status register.
typedef enum eQUES_TEMP_STATUS
{
	eQUES_TEMP_STATUS_OVT			// Over temperature protection.
}teQUES_TEMP_STATUS;

/*---- Mask Flags for use with return values from DC850W_QueryRegister Function ----*/ 

// Operation Register Masks
#define AMETEKDC_MASK_OPER_CAL	((ViInt16)(1<<eOPER_STATUS_CAL))
#define AMETEKDC_MASK_MEAS     ((ViInt16)(1<<eOPER_STATUS_MEAS))
#define AMETEKDC_MASK_SD     ((ViInt16)(1<<eOPER_STATUS_SD))
#define AMETEKDC_MASK_LLC     ((ViInt16)(1<<eOPER_STATUS_LLOC))
#define AMETEKDC_MASK_CSH     ((ViInt16)(1<<eOPER_STATUS_CSHARE))
#define AMETEKDC_MASK_CV     ((ViInt16)(1<<eOPER_STATUS_CV))      /* Constant Voltage */
#define AMETEKDC_MASK_CC     ((ViInt16)(1<<eOPER_STATUS_CC))      /* Constant Current */
#define AMETEKDC_MASK_PRG     ((ViInt16)(1<<eOPER_STATUS_PROG_RUN))

// Operation Register Current Share Masks
#define AMETEKDC_MASK_MST     ((ViInt16)(1<<eOPER_CSH_SREG_MASTER))
#define AMETEKDC_MASK_SLV     ((ViInt16)(1<<eOPER_CSH_SREG_SLAVE))	

// Operation Register Shutdown Masks
#define AMETEKDC_MASK_PRT     ((ViInt16)(1<<eOPER_SD_STATUS_PROT))
#define AMETEKDC_MASK_INT     ((ViInt16)(1<<eOPER_SD_STATUS_INT))
#define AMETEKDC_MASK_XSD     ((ViInt16)(1<<eOPER_SD_STATUS_EXTSD))

// Operation Register Shutdown Protection Masks
#define AMETEKDC_MASK_OVP     ((ViInt16)(1<<eOPER_SD_PROT_SREG_OVP))      /* Over Voltage */
#define AMETEKDC_MASK_UVP     ((ViInt16)(1<<eOPER_SD_PROT_SREG_UVP))      /* Under Voltage */
#define AMETEKDC_MASK_OCP     ((ViInt16)(1<<eOPER_SD_PROT_SREG_OCP))      /* Over Current */
#define AMETEKDC_MASK_AC     ((ViInt16)(1<<eOPER_SD_PROT_SREG_ACOFF))
#define AMETEKDC_MASK_OTP     ((ViInt16)(1<<eQUES_TEMP_STATUS_OVT))      /* Over Temperature */
#define AMETEKDC_MASK_FLD     ((ViInt16)(1<<eOPER_SD_PROT_SREG_FLD))
#define AMETEKDC_MASK_FAN     ((ViInt16)(1<<eOPER_SD_PROT_SREG_FAN))

// Questionable Register Masks
#define AMETEKDC_MASK_VOLT     ((ViInt16)(1<<eQUES_STATUS_VOLT))
#define AMETEKDC_MASK_TEMP     ((ViInt16)(1<<eQUES_STATUS_TEMP))
#define AMETEKDC_MASK_CAL     ((ViInt16)(1<<eQUES_STATUS_CAL))
#define AMETEKDC_MASK_ACOFF   ((ViInt16)(1<<eQUES_STATUS_AC_OFF))

// Questionable Register Voltage Masks
#define AMETEKDC_MASK_OVV   ((ViInt16)(1<<eQUES_VOLT_STATUS_OVV))
#define AMETEKDC_MASK_UDV   ((ViInt16)(1<<eQUES_VOLT_STATUS_UDV))

// Questionable Register Temperature Masks
#define AMETEKDC_MASK_OVT   ((ViInt16)(1<<eQUES_TEMP_STATUS_OVT))

/**************************************************************************** 
 *------------------------------ Useful Macros -----------------------------* 
 ****************************************************************************/

/*- Defined values for rangeType parameter of function -----------------*/
/*- ConfigureOutputRange -----------------------------------------------*/
#define AMETEKDC_VAL_RANGE_CURRENT               IVIDCPWR_VAL_RANGE_CURRENT              
#define AMETEKDC_VAL_RANGE_VOLTAGE               IVIDCPWR_VAL_RANGE_VOLTAGE

/*- Defined values for outputState parameter of function ---------------*/
/*- QueryOutputState ---------------------------------------------------*/
#define AMETEKDC_VAL_OUTPUT_CONSTANT_VOLTAGE     IVIDCPWR_VAL_OUTPUT_CONSTANT_VOLTAGE
#define AMETEKDC_VAL_OUTPUT_CONSTANT_CURRENT     IVIDCPWR_VAL_OUTPUT_CONSTANT_CURRENT
//#define AMETEKDC_VAL_OUTPUT_CONSTANT_POWER      (IVIDCPWR_VAL_OUTPUT_STATE_SPECIFIC_EXT_BASE + 1L)
#define AMETEKDC_VAL_OUTPUT_UNREGULATED          IVIDCPWR_VAL_OUTPUT_UNREGULATED
#define AMETEKDC_VAL_OUTPUT_OVER_VOLTAGE         IVIDCPWR_VAL_OUTPUT_OVER_VOLTAGE       
#define AMETEKDC_VAL_OUTPUT_OVER_CURRENT         IVIDCPWR_VAL_OUTPUT_OVER_CURRENT       
//#define AMETEKDC_VAL_OUTPUT_OVER_POWER          (IVIDCPWR_VAL_OUTPUT_STATE_SPECIFIC_EXT_BASE + 2L)
#define AMETEKDC_VAL_OUTPUT_OVER_TEMPERATURE    (IVIDCPWR_VAL_OUTPUT_STATE_SPECIFIC_EXT_BASE + 3L)
#define AMETEKDC_VAL_OUTPUT_UNDER_VOLTAGE       (IVIDCPWR_VAL_OUTPUT_STATE_SPECIFIC_EXT_BASE + 4L)
//#define AMETEKDC_VAL_OUTPUT_UNDER_CURRENT       (IVIDCPWR_VAL_OUTPUT_STATE_SPECIFIC_EXT_BASE + 5L)
//#define AMETEKDC_VAL_OUTPUT_UNDER_POWER         (IVIDCPWR_VAL_OUTPUT_STATE_SPECIFIC_EXT_BASE + 6L)
//#define AMETEKDC_VAL_OUTPUT_HIGH_TEMPERATURE    (IVIDCPWR_VAL_OUTPUT_STATE_SPECIFIC_EXT_BASE + 7L)

/*- Defined values for measurementType parameter of function -----------*/
/*- Measure ------------------------------------------------------------*/
#define AMETEKDC_VAL_MEASURE_CURRENT                   IVIDCPWR_VAL_MEASURE_CURRENT
#define AMETEKDC_VAL_MEASURE_VOLTAGE                   IVIDCPWR_VAL_MEASURE_VOLTAGE
#define AMETEKDC_VAL_MEASURE_ANALOG_VOLT_INP    (IVIDCPWR_VAL_MEASURE_SPECIFIC_EXT_BASE + 0L)
#define AMETEKDC_VAL_MEASURE_ANALOG_CURR_INP    (IVIDCPWR_VAL_MEASURE_SPECIFIC_EXT_BASE + 1L) 
#define AMETEKDC_VAL_MEASURE_ISOL_ANALOG_VOLT_INP    (IVIDCPWR_VAL_MEASURE_SPECIFIC_EXT_BASE + 2L)
#define AMETEKDC_VAL_MEASURE_ISOL_ANALOG_CURR_INP    (IVIDCPWR_VAL_MEASURE_SPECIFIC_EXT_BASE + 3L) 


/*- Defined values for Behavior parameter of function ------------------*/
/*- ConfigureProtectionSetpoint ----------------------------------------*/
#define AMETEKDC_VAL_PROTECTION_WARNING     0L   
#define AMETEKDC_VAL_PROTECTION_TRIP        1L  

/*- Defined values for State parameter of function ---------------------*/
/*- ChangeAutoSequenceState ----------------------------------------*/
#define AMETEKDC_VAL_RUN                    0L   
#define AMETEKDC_VAL_PAUSE                  1L   
#define AMETEKDC_VAL_STOP                   2L  

/*- Defined values for Protection Type parameter of function -----------*/
/*- QueryProtectionTripped ---------------------------------------------*/
#define AMETEKDC_VAL_OVP                    0L   
#define AMETEKDC_VAL_OCP                    1L 
#define AMETEKDC_VAL_UVP                    2L   
#define AMETEKDC_VAL_SD                     4L   
#define AMETEKDC_VAL_FAN                    5L  
#define AMETEKDC_VAL_FOLDBACK               6L   
#define AMETEKDC_VAL_OVER_TEMP              7L   
#define AMETEKDC_VAL_AC_FAIL                8L  
#define AMETEKDC_VAL_INTERLOCK			  9L
#define AMETEKDC_VAL_FAN_SLAVE		10L
#define AMETEKDC_VAL_OVER_TEMP_SLAVE	11L
#define AMETEKDC_VAL_OCP_SLAVE		12L


/*- Defined values for SCPI Command Lookup table  ---------*/
// Note: to facilitate error checking the status register and 
// measurement definitions must be contiguous and maintain the 
// order in this set of definitions. Failure to do so may cause new 
// commands to be reject by the error checking of the queryRegister 
// and setRegister functions.  When a new command is added the 
// definitions on first and last command should be update if the 
// new command causes this to be changed.
#define AMETEKDC_VAL_SYST_FPANEL                                0L
#define AMETEKDC_VAL_SYST_PROT_MASK                             1L
#define AMETEKDC_VAL_SYST_PROT_LATCH                            2L
#define AMETEKDC_VAL_SYST_ERROR                                 3L
#define AMETEKDC_VAL_SYST_ERROR_CODE                            4L
#define AMETEKDC_VAL_SYST_ERROR_COUNT                           5L
#define AMETEKDC_VAL_SYST_VERS                                  6L
#define AMETEKDC_VAL_SYST_SAVE                                  7L
#define AMETEKDC_VAL_SYST_REC                                   8L
#define AMETEKDC_VAL_SYST_RES                                   9L
#define AMETEKDC_VAL_SYST_WAIT                                 10L
#define AMETEKDC_VAL_SYST_TEST                                 11L
#define AMETEKDC_VAL_SYST_IDEN                                 12L
#define AMETEKDC_VAL_SYST_REMOTE_STATE                         13L
#define AMETEKDC_VAL_SYST_REMOTE_SOUR_VOLT                     14L
#define AMETEKDC_VAL_SYST_REMOTE_SOUR_CURR                     15L
#define AMETEKDC_VAL_SYST_COMM_ADR                             16L
#define AMETEKDC_VAL_SYST_COMM_VAP_LEVEL                       17L
#define AMETEKDC_VAL_SYST_COMM_CAP_LEVEL                       18L
#define AMETEKDC_VAL_SYST_HELP                                 19L
#define AMETEKDC_VAL_SYST_HELP_LEGACY                          20L 
#define AMETEKDC_VAL_SYST_HELP_SYNTAX                          21L
#define AMETEKDC_VAL_STAT_PRESET                               22L
#define AMETEKDC_VAL_STAT_CLEAR                                23L
#define AMETEKDC_VAL_STAT_QUEST_EVENT                          24L
#define AMETEKDC_VAL_STAT_QUEST_ENABLE                         25L
#define AMETEKDC_VAL_STAT_QUEST_COND                           26L
#define AMETEKDC_VAL_STAT_QUEST_PTR                            27L
#define AMETEKDC_VAL_STAT_QUEST_NTR                            28L
#define AMETEKDC_VAL_STAT_QUEST_VOLT_EVENT                     29L
#define AMETEKDC_VAL_STAT_QUEST_VOLT_ENABLE                    30L
#define AMETEKDC_VAL_STAT_QUEST_VOLT_COND                      31L
#define AMETEKDC_VAL_STAT_QUEST_VOLT_PTR                       32L
#define AMETEKDC_VAL_STAT_QUEST_VOLT_NTR                       33L
#define AMETEKDC_VAL_STAT_QUEST_TEMP_EVENT                     34L
#define AMETEKDC_VAL_STAT_QUEST_TEMP_ENABLE                    35L
#define AMETEKDC_VAL_STAT_QUEST_TEMP_COND                      36L
#define AMETEKDC_VAL_STAT_QUEST_TEMP_PTR                       37L
#define AMETEKDC_VAL_STAT_QUEST_TEMP_NTR                       38L
#define AMETEKDC_VAL_STAT_OPER_EVENT                           39L
#define AMETEKDC_VAL_STAT_OPER_ENABLE                          40L
#define AMETEKDC_VAL_STAT_OPER_COND                            41L
#define AMETEKDC_VAL_STAT_OPER_PTR                             42L
#define AMETEKDC_VAL_STAT_OPER_NTR                             43L
#define AMETEKDC_VAL_STAT_OPER_CSH_EVENT                       44L
#define AMETEKDC_VAL_STAT_OPER_CSH_ENABLE                      45L
#define AMETEKDC_VAL_STAT_OPER_CSH_COND                        46L
#define AMETEKDC_VAL_STAT_OPER_CSH_PTR                         47L
#define AMETEKDC_VAL_STAT_OPER_CSH_NTR                         48L
#define AMETEKDC_VAL_STAT_OPER_SHUT_EVENT                      49L
#define AMETEKDC_VAL_STAT_OPER_SHUT_ENABLE                     50L
#define AMETEKDC_VAL_STAT_OPER_SHUT_COND                       51L
#define AMETEKDC_VAL_STAT_OPER_SHUT_PTR                        52L
#define AMETEKDC_VAL_STAT_OPER_SHUT_NTR                        53L
#define AMETEKDC_VAL_STAT_OPER_SHUT_PROT_EVENT                 54L
#define AMETEKDC_VAL_STAT_OPER_SHUT_PROT_ENABLE                55L
#define AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND                  56L
#define AMETEKDC_VAL_STAT_OPER_SHUT_PROT_PTR                   57L
#define AMETEKDC_VAL_STAT_OPER_SHUT_PROT_NTR                   58L
#define AMETEKDC_VAL_STAT_STANDARD_EVENT                       59L
#define AMETEKDC_VAL_STAT_STANDARD_ENABLE                      60L
#define AMETEKDC_VAL_STAT_STATUS_BYTE                          61L
#define AMETEKDC_VAL_STAT_SERVICE_REQUEST                      62L
#define AMETEKDC_VAL_SENS_PROT_INTERLOCK                       63L
#define AMETEKDC_VAL_SOUR_VOLT                                 64L
#define AMETEKDC_VAL_SOUR_VOLT_PROT_OVP                        65L
#define AMETEKDC_VAL_SOUR_VOLT_PROT_UVP                        66L
#define AMETEKDC_VAL_SOUR_CURR                                 67L
#define AMETEKDC_VAL_SOUR_CURR_PROT                            68L
#define AMETEKDC_VAL_SOUR_COMB_CSH                             69L
#define AMETEKDC_VAL_PROG_STATE                                70L
#define AMETEKDC_VAL_PROG_START_RECORDING                      71L
#define AMETEKDC_VAL_PROG_STOP_RECORDING                       72L
#define AMETEKDC_VAL_PROG_DELETE_ALL                           73L
#define AMETEKDC_VAL_PROG_REPEAT                               74L
#define AMETEKDC_VAL_PROG_DWELL                                75L
#define AMETEKDC_VAL_PROG_READBACK                             76L
#define AMETEKDC_VAL_OUTP_PROT_CLEAR                           77L
#define AMETEKDC_VAL_OUTP_PROT_FOLD                            78L
#define AMETEKDC_VAL_OUTP_PROT_FOLD_DELAY                      79L
#define AMETEKDC_VAL_OUTP_SD_POL                               80L
#define AMETEKDC_VAL_OUTP_STATE                                81L
#define AMETEKDC_VAL_OUTP_PON                                  82L
#define AMETEKDC_VAL_OUTP_AUX                                  83L
#define AMETEKDC_VAL_OUTP_AUX_PON                              84L
#define AMETEKDC_VAL_MEAS_VOLT                                 85L
#define AMETEKDC_VAL_MEAS_CURR                                 86L
#define AMETEKDC_VAL_MEAS_VAP                                  87L
#define AMETEKDC_VAL_MEAS_VAP_ISOL                             88L
#define AMETEKDC_VAL_MEAS_CAP                                  89L
#define AMETEKDC_VAL_MEAS_CAP_ISOL                             90L
#define AMETEKDC_VAL_INITIATE_IMM                              91L
#define AMETEKDC_VAL_CAL_RESTORE                               92L
#define AMETEKDC_VAL_CAL_OVP                                   93L
#define AMETEKDC_VAL_CAL_PARAM_INPUT                           94L
#define AMETEKDC_VAL_CAL_PARAM                                 95L
#define AMETEKDC_VAL_CAL_PARAM_APR                             96L
#define AMETEKDC_VAL_CAL_OUTP_VOLT                             97L
#define AMETEKDC_VAL_CAL_OUTP_CURR                             98L
#define AMETEKDC_VAL_CAL_OUTP_VAP_VOLT                         99L
#define AMETEKDC_VAL_CAL_OUTP_VAP_VOLT_ISOL                   100L
#define AMETEKDC_VAL_CAL_OUTP_CAP_VOLT                        101L
#define AMETEKDC_VAL_CAL_OUTP_CAP_VOLT_ISOL                   102L
#define AMETEKDC_VAL_CAL_OUTP_VAP_RES                         103L
#define AMETEKDC_VAL_CAL_OUTP_VAP_RES_ISOL                    104L
#define AMETEKDC_VAL_CAL_OUTP_CAP_RES                         105L
#define AMETEKDC_VAL_CAL_OUTP_CAP_RES_ISOL                    106L
#define AMETEKDC_VAL_CAL_MONITOR_VOLT                         107L
#define AMETEKDC_VAL_CAL_MONITOR_VOLT_ISOL                    108L
#define AMETEKDC_VAL_CAL_MONITOR_CURR                         109L
#define AMETEKDC_VAL_CAL_MONITOR_CURR_ISOL                    110L
#define AMETEKDC_VAL_CAL_INPUT_VAP                            111L
#define AMETEKDC_VAL_CAL_INPUT_CAP                            112L
#define AMETEKDC_VAL_SYST_TRG                                 113L

// As noted about these two definitions are necessary for error checking in
// the queryRegister and setRegister functions.
#define AMETEKDC_VAL_FIRST_REGISTER_CMD     AMETEKDC_VAL_STAT_QUEST_EVENT
#define AMETEKDC_VAL_LAST_REGISTER_CMD      AMETEKDC_VAL_STAT_SERVICE_REQUEST
#define AMETEKDC_VAL_FIRST_MEASURE_CMD     AMETEKDC_VAL_MEAS_VOLT
#define AMETEKDC_VAL_LAST_MEASURE_CMD      AMETEKDC_VAL_MEAS_CAP_ISOL

/*- Defined values for queryRegister parameter of function -------------*/
/*- QueryRegister & SetRegister functions ---------------------------*/
        
#define AMETEKDC_VAL_STD_EVENT_STATUS_ENABLE    AMETEKDC_VAL_STAT_STANDARD_ENABLE
#define AMETEKDC_VAL_STD_EVENT_STATUS_EVENT                 AMETEKDC_VAL_STAT_STANDARD_EVENT
#define AMETEKDC_VAL_SERVICE_REQUEST_ENABLE                AMETEKDC_VAL_STAT_SERVICE_REQUEST
#define AMETEKDC_VAL_STATUS_BYTE                                    AMETEKDC_VAL_STAT_STATUS_BYTE

#define AMETEKDC_VAL_OPERATION_STATUS_EVENT                 AMETEKDC_VAL_STAT_OPER_EVENT
#define AMETEKDC_VAL_OPERATION_STATUS_CONDITION          AMETEKDC_VAL_STAT_OPER_COND
#define AMETEKDC_VAL_OPERATION_STATUS_ENABLE               AMETEKDC_VAL_STAT_OPER_ENABLE
#define AMETEKDC_VAL_OPERATION_STATUS_PTRANSITION       AMETEKDC_VAL_STAT_OPER_PTR
#define AMETEKDC_VAL_OPERATION_STATUS_NTRANSITION       AMETEKDC_VAL_STAT_OPER_NTR

#define AMETEKDC_VAL_OPERATION_SHUTDOWN_EVENT            AMETEKDC_VAL_STAT_OPER_SHUT_EVENT
#define AMETEKDC_VAL_OPERATION_SHUTDOWN_CONDITION     AMETEKDC_VAL_STAT_OPER_SHUT_COND
#define AMETEKDC_VAL_OPERATION_SHUTDOWN_ENABLE          AMETEKDC_VAL_STAT_OPER_SHUT_ENABLE
#define AMETEKDC_VAL_OPERATION_SHUTDOWN_PTRANSITION  AMETEKDC_VAL_STAT_OPER_SHUT_PTR
#define AMETEKDC_VAL_OPERATION_SHUTDOWN_NTRANSITION  AMETEKDC_VAL_STAT_OPER_SHUT_NTR

#define AMETEKDC_VAL_OPERATION_PROTECTION_EVENT           AMETEKDC_VAL_STAT_OPER_SHUT_PROT_EVENT
#define AMETEKDC_VAL_OPERATION_PROTECTION_CONDITION    AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND
#define AMETEKDC_VAL_OPERATION_PROTECTION_ENABLE         AMETEKDC_VAL_STAT_OPER_SHUT_PROT_ENABLE
#define AMETEKDC_VAL_OPERATION_PROTECTION_PTRANSITION AMETEKDC_VAL_STAT_OPER_SHUT_PROT_PTR
#define AMETEKDC_VAL_OPERATION_PROTECTION_NTRANSITION AMETEKDC_VAL_STAT_OPER_SHUT_PROT_NTR

#define AMETEKDC_VAL_OPERATION_CSHARE_EVENT                  AMETEKDC_VAL_STAT_OPER_CSH_EVENT
#define AMETEKDC_VAL_OPERATION_CSHARE_CONDITION           AMETEKDC_VAL_STAT_OPER_CSH_COND
#define AMETEKDC_VAL_OPERATION_CSHARE_ENABLE                AMETEKDC_VAL_STAT_OPER_CSH_ENABLE
#define AMETEKDC_VAL_OPERATION_CSHARE_PTRANSITION        AMETEKDC_VAL_STAT_OPER_CSH_PTR
#define AMETEKDC_VAL_OPERATION_CSHARE_NTRANSITION        AMETEKDC_VAL_STAT_OPER_CSH_NTR

#define AMETEKDC_VAL_QUESTIONABLE_STATUS_EVENT             AMETEKDC_VAL_STAT_QUEST_EVENT
#define AMETEKDC_VAL_QUESTIONABLE_STATUS_CONDITION      AMETEKDC_VAL_STAT_QUEST_COND
#define AMETEKDC_VAL_QUESTIONABLE_STATUS_ENABLE           AMETEKDC_VAL_STAT_QUEST_ENABLE
#define AMETEKDC_VAL_QUESTIONABLE_STATUS_PTRANSITION   AMETEKDC_VAL_STAT_QUEST_PTR
#define AMETEKDC_VAL_QUESTIONABLE_STATUS_NTRANSITION   AMETEKDC_VAL_STAT_QUEST_NTR

#define AMETEKDC_VAL_QUESTIONABLE_VOLTAGE_EVENT           AMETEKDC_VAL_STAT_QUEST_VOLT_EVENT
#define AMETEKDC_VAL_QUESTIONABLE_VOLTAGE_CONDITION    AMETEKDC_VAL_STAT_QUEST_VOLT_COND
#define AMETEKDC_VAL_QUESTIONABLE_VOLTAGE_ENABLE         AMETEKDC_VAL_STAT_QUEST_VOLT_ENABLE
#define AMETEKDC_VAL_QUESTIONABLE_VOLTAGE_PTRANSITION AMETEKDC_VAL_STAT_QUEST_VOLT_PTR
#define AMETEKDC_VAL_QUESTIONABLE_VOLTAGE_NTRANSITION AMETEKDC_VAL_STAT_QUEST_VOLT_NTR

#define AMETEKDC_VAL_QUESTIONABLE_TEMPERATURE_EVENT            AMETEKDC_VAL_STAT_QUEST_TEMP_EVENT
#define AMETEKDC_VAL_QUESTIONABLE_TEMPERATURE_CONDITION     AMETEKDC_VAL_STAT_QUEST_TEMP_COND
#define AMETEKDC_VAL_QUESTIONABLE_TEMPERATURE_ENABLE          AMETEKDC_VAL_STAT_QUEST_TEMP_ENABLE
#define AMETEKDC_VAL_QUESTIONABLE_TEMPERATURE_PTRANSITION  AMETEKDC_VAL_STAT_QUEST_TEMP_PTR
#define AMETEKDC_VAL_QUESTIONABLE_TEMPERATURE_NTRANSITION  AMETEKDC_VAL_STAT_QUEST_TEMP_NTR

    
/**************************************************************************** 
 *---------------------------- Attribute Defines ---------------------------* 
 ****************************************************************************/

/*- IVI Inherent Instrument Attributes ---------------------------------*/    

        /* User Options */
#define AMETEKDC_ATTR_RANGE_CHECK                    IVI_ATTR_RANGE_CHECK                    /* ViBoolean */
#define AMETEKDC_ATTR_QUERY_INSTRUMENT_STATUS        IVI_ATTR_QUERY_INSTRUMENT_STATUS        /* ViBoolean */
#define AMETEKDC_ATTR_CACHE                          IVI_ATTR_CACHE                          /* ViBoolean */
#define AMETEKDC_ATTR_SIMULATE                       IVI_ATTR_SIMULATE                       /* ViBoolean */
#define AMETEKDC_ATTR_RECORD_COERCIONS               IVI_ATTR_RECORD_COERCIONS               /* ViBoolean */

#define AMETEKDC_ATTR_DRIVER_SETUP                   IVI_ATTR_DRIVER_SETUP                   /* ViString */
#define AMETEKDC_ATTR_INTERCHANGE_CHECK              IVI_ATTR_INTERCHANGE_CHECK              /* ViBoolean */

        /* Instrument Capabilities */
#define AMETEKDC_ATTR_CHANNEL_COUNT                  IVI_ATTR_CHANNEL_COUNT                  /* ViInt32,  read-only */
#define AMETEKDC_ATTR_GROUP_CAPABILITIES             IVI_ATTR_GROUP_CAPABILITIES             /* ViString, read-only */

        /* Driver Information  */
#define AMETEKDC_ATTR_SPECIFIC_DRIVER_PREFIX         IVI_ATTR_SPECIFIC_DRIVER_PREFIX         /* ViString, read-only  */
#define AMETEKDC_ATTR_SUPPORTED_INSTRUMENT_MODELS    IVI_ATTR_SUPPORTED_INSTRUMENT_MODELS    /* ViString, read-only  */
#define AMETEKDC_ATTR_INSTRUMENT_MANUFACTURER        IVI_ATTR_INSTRUMENT_MANUFACTURER        /* ViString, read-only  */
#define AMETEKDC_ATTR_INSTRUMENT_MODEL               IVI_ATTR_INSTRUMENT_MODEL               /* ViString, read-only  */
#define AMETEKDC_ATTR_INSTRUMENT_FIRMWARE_REVISION   IVI_ATTR_INSTRUMENT_FIRMWARE_REVISION   /* ViString, read-only  */
#define AMETEKDC_ATTR_SPECIFIC_DRIVER_REVISION       IVI_ATTR_SPECIFIC_DRIVER_REVISION       /* ViString, read-only  */
#define AMETEKDC_ATTR_SPECIFIC_DRIVER_VENDOR         IVI_ATTR_SPECIFIC_DRIVER_VENDOR         /* ViString, read-only  */
#define AMETEKDC_ATTR_SPECIFIC_DRIVER_DESCRIPTION    IVI_ATTR_SPECIFIC_DRIVER_DESCRIPTION    /* ViString, read-only  */
#define AMETEKDC_ATTR_SPECIFIC_DRIVER_CLASS_SPEC_MAJOR_VERSION IVI_ATTR_SPECIFIC_DRIVER_CLASS_SPEC_MAJOR_VERSION /* ViInt32, read-only */
#define AMETEKDC_ATTR_SPECIFIC_DRIVER_CLASS_SPEC_MINOR_VERSION IVI_ATTR_SPECIFIC_DRIVER_CLASS_SPEC_MINOR_VERSION /* ViInt32, read-only */

        /* Advanced Session Information */
#define AMETEKDC_ATTR_LOGICAL_NAME                IVI_ATTR_LOGICAL_NAME                   /* ViString, read-only  */
#define AMETEKDC_ATTR_IO_RESOURCE_DESCRIPTOR      IVI_ATTR_IO_RESOURCE_DESCRIPTOR         /* ViString, read-only  */

        /*- Basic Instrument Operations ----------------------------------------*/
#define AMETEKDC_ATTR_VOLTAGE_LEVEL               IVIDCPWR_ATTR_VOLTAGE_LEVEL           /* ViReal64,  multi-channel, volts */
#define AMETEKDC_ATTR_OVP_LIMIT                   IVIDCPWR_ATTR_OVP_LIMIT               /* ViReal64,  multi-channel, volts */
#define AMETEKDC_ATTR_CURRENT_LIMIT               IVIDCPWR_ATTR_CURRENT_LIMIT           /* ViReal64,  multi-channel, amps  */
#define AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR      IVIDCPWR_ATTR_CURRENT_LIMIT_BEHAVIOR  /* ViInt32,   multi-channel */
#define AMETEKDC_ATTR_OUTPUT_ENABLED              IVIDCPWR_ATTR_OUTPUT_ENABLED          /* ViBoolean, multi-channel        */

  
       /*- Instrument-Specific Attributes -------------------------------------*/ 
#define AMETEKDC_ATTR_ID_QUERY_RESPONSE              (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 1L)     /* ViString(Read Only) */
#define AMETEKDC_ATTR_FOLDBACK_SETTING               (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 7L)
#define AMETEKDC_ATTR_FOLD_DELAY                     (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 15L)
#define AMETEKDC_ATTR_OUTPUT_ENABLED_POWER_ON        (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 17L)
#define AMETEKDC_ATTR_OCP_LIMIT                      (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 20L)
#define AMETEKDC_ATTR_UVP_LIMIT                      (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 24L)
#define AMETEKDC_ATTR_UVP_BEHAVIOR                   (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 28L)
#define AMETEKDC_ATTR_CURRENT_SHARE_MODE             (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 29L)
#define AMETEKDC_ATTR_OVER_TEMPERATURE_RESPONSE      (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 30L)
#define AMETEKDC_ATTR_AC_FAIL_RESPONSE               (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 31L) 
#define AMETEKDC_ATTR_REMOTE_MODE                    (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 32L)
#define AMETEKDC_ATTR_VOLT_ANALOG_PROGRAMMING_RANGE (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 33L)
#define AMETEKDC_ATTR_REMOTE_SOURCE_VOLT (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 36L)
#define AMETEKDC_ATTR_MODEL_TYPE                     (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 37L)
#define AMETEKDC_ATTR_MULTICHANNEL_FIRMWARE_REVISION (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 39L)
#define AMETEKDC_ATTR_CONFIGURABLE_DELAY             (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 40L) 
#define AMETEKDC_ATTR_OVER_TEMPERATURE_SLAVE_RESPONSE (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 41L)
#define AMETEKDC_ATTR_FOLDBACK_RESPONSE       (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 42L)
#define AMETEKDC_ATTR_CURR_ANALOG_PROGRAMMING_RANGE (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 43L)
#define AMETEKDC_ATTR_REMOTE_SOURCE_CURR      (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 44L)
#define AMETEKDC_ATTR_FP_INACTIVITY_TIMEOUT   (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 45L)
#define AMETEKDC_ATTR_AUX_ENABLED             (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 46L)
#define AMETEKDC_ATTR_AUX_ENABLED_POWER_ON    (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 47L)
#define AMETEKDC_ATTR_EXT_SHUTDOWN_LOGIC      (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 48L)
#define AMETEKDC_ATTR_AUTO_SEQUENCE_DWELL_TIME (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 49L)
#define AMETEKDC_ATTR_PROTECTION_MASK         (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 50L)
#define AMETEKDC_ATTR_DATARATE                (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 51L)
#define AMETEKDC_ATTR_BUS_ADDRESS             (IVI_SPECIFIC_PUBLIC_ATTR_BASE + 52L)
#define AMETEKDC_ATTR_OVP_ENABLED             IVIDCPWR_ATTR_OVP_ENABLED

        /*- Trigger Attributes -------------------------------------------------*/
    
/**************************************************************************** 
 *------------------------ Attribute Value Defines -------------------------* 
 ****************************************************************************/
  
/*- Defined values for attribute AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR ---*/
#define AMETEKDC_VAL_CURRENT_REGULATE       IVIDCPWR_VAL_CURRENT_REGULATE
#define AMETEKDC_VAL_CURRENT_TRIP           IVIDCPWR_VAL_CURRENT_TRIP

/*- Defined values for attribute AMETEKDC_ATTR_REMOTE_SOURCE ------------*/
//#define AMETEKDC_VAL_LOC	                  0L   // Commented to avoid redefining below.
#define AMETEKDC_VAL_IAV                    1L
#define AMETEKDC_VAL_IAR                    2L
#define AMETEKDC_VAL_AVOL                   3L
#define AMETEKDC_VAL_ARES                   4L

/*- Defined values for attribute AMETEKDC_ATTR_REMOTE_CONTROL_OPERATION -*/
#define AMETEKDC_VAL_LOC                    0L
#define AMETEKDC_VAL_REM                    1L


/*- Defined values for attribute AMETEKDC_ATTR_FOLDBACK_SETTING ---------*/
#define AMETEKDC_VAL_FOLD_NONE              0L
#define AMETEKDC_VAL_FOLD_CC                1L
#define AMETEKDC_VAL_FOLD_CV                2L



/**************************************************************************** 
 *---------------- Instrument Driver Function Declarations -----------------* 
 ****************************************************************************/

/*------- START: Ivi_DCPwr Required Utility Functions -------------------------------*/
ViStatus _VI_FUNC AmetekDC_init( ViRsrc resourceName, ViBoolean IDQuery,
                               ViBoolean resetDevice, ViSession *vi );

ViStatus _VI_FUNC AmetekDC_close( ViSession vi );


ViStatus _VI_FUNC AmetekDC_reset( ViSession vi );


ViStatus _VI_FUNC AmetekDC_self_test( ViSession vi, ViInt16 *selfTestResult,
                                    ViChar selfTestMessage[],ViConstString channelName);							

ViStatus _VI_FUNC AmetekDC_error_query( ViSession vi, ViInt32 *errCode,
                                      ViChar errorMessage[] );                                          

ViStatus _VI_FUNC AmetekDC_error_message( ViSession vi, ViStatus errorCode,
                                        ViChar errorMessage[256] ); 


ViStatus _VI_FUNC AmetekDC_revision_query( ViSession vi, 
                                         ViChar instrumentDriverRevision[],
                                         ViChar firmwareRevision[] );

	/*- Utility Functions -*/
ViStatus _VI_FUNC AmetekDC_InvalidateAllAttributes( ViSession vi );

ViStatus _VI_FUNC AmetekDC_ResetWithDefaults( ViSession vi );

ViStatus _VI_FUNC AmetekDC_Disable( ViSession vi );


	/*- Init and Close Functions -*/                               
ViStatus _VI_FUNC AmetekDC_InitWithOptions( ViRsrc resourceName, 
                                          ViBoolean IDQuery,
                                          ViBoolean resetDevice, 
                                          ViChar optionString[], 
                                          ViSession *newVi );

	/*- Set, Get, and Check Attribute Functions -*/
ViStatus _VI_FUNC  AmetekDC_GetAttributeViInt32( ViSession vi, ViConstString channelName, ViAttr attribute, ViInt32 *value );
ViStatus _VI_FUNC  AmetekDC_GetAttributeViReal64( ViSession vi, ViConstString channelName, ViAttr attribute, ViReal64 *value );
ViStatus _VI_FUNC  AmetekDC_GetAttributeViString( ViSession vi, ViConstString channelName, ViAttr attribute, ViInt32 bufSize, ViChar value[] ); 
ViStatus _VI_FUNC  AmetekDC_GetAttributeViSession( ViSession vi, ViConstString channelName, ViAttr attribute, ViSession *value );
ViStatus _VI_FUNC  AmetekDC_GetAttributeViBoolean( ViSession vi, ViConstString channelName, ViAttr attribute, ViBoolean *value );

ViStatus _VI_FUNC  AmetekDC_SetAttributeViInt32( ViSession vi, ViConstString channelName, ViAttr attribute, ViInt32 value );
ViStatus _VI_FUNC  AmetekDC_SetAttributeViReal64( ViSession vi, ViConstString channelName, ViAttr attribute, ViReal64 value );
ViStatus _VI_FUNC  AmetekDC_SetAttributeViString( ViSession vi, ViConstString channelName, ViAttr attribute, ViConstString value ); 
ViStatus _VI_FUNC  AmetekDC_SetAttributeViSession( ViSession vi, ViConstString channelName, ViAttr attribute, ViSession value );
ViStatus _VI_FUNC  AmetekDC_SetAttributeViBoolean( ViSession vi, ViConstString channelName, ViAttr attribute, ViBoolean value );

ViStatus _VI_FUNC  AmetekDC_CheckAttributeViInt32( ViSession vi, ViConstString channelName, ViAttr attribute, ViInt32 value );
ViStatus _VI_FUNC  AmetekDC_CheckAttributeViReal64( ViSession vi, ViConstString channelName, ViAttr attribute, ViReal64 value );
ViStatus _VI_FUNC  AmetekDC_CheckAttributeViString( ViSession vi, ViConstString channelName, ViAttr attribute, ViConstString value ); 
ViStatus _VI_FUNC  AmetekDC_CheckAttributeViSession( ViSession vi, ViConstString channelName, ViAttr attribute, ViSession value );
ViStatus _VI_FUNC  AmetekDC_CheckAttributeViBoolean( ViSession vi, ViConstString channelName, ViAttr attribute, ViBoolean value );


	/*- Locking Functions -*/
ViStatus _VI_FUNC AmetekDC_LockSession( ViSession vi, ViBoolean *callerHasLock );

ViStatus _VI_FUNC AmetekDC_UnlockSession( ViSession vi, ViBoolean *callerHasLock );


	/*- Error Information Functions -*/                                         
ViStatus _VI_FUNC AmetekDC_GetError( ViSession vi, ViStatus *errorCode, 
                                   ViInt32 bufferSize,
                                   ViChar description[] );
                                          
ViStatus _VI_FUNC AmetekDC_ClearError( ViSession vi );


	/*- Interchangeability Info Functions -*/
ViStatus _VI_FUNC AmetekDC_GetNextInterchangeWarning( ViSession vi, 
                                                    ViInt32 bufferSize, 
                                                    ViChar warnString[] );

ViStatus _VI_FUNC AmetekDC_ClearInterchangeWarnings( ViSession vi );
ViStatus _VI_FUNC AmetekDC_ResetInterchangeCheck( ViSession vi );

ViStatus _VI_FUNC AmetekDC_GetNextCoercionRecord( ViSession vi, ViInt32 bufferSize,
                                                ViChar record[] );  

	/*- IviDCPwr Basic Operation Functions -*/
ViStatus _VI_FUNC AmetekDC_ConfigureOutputEnabled( ViSession vi,
                                                 ViConstString channelName,
                                                 ViBoolean enabled );

ViStatus _VI_FUNC AmetekDC_ConfigureOutputRange( ViSession vi, 
                                               ViConstString channelName, 
                                               ViInt32 rangeType, 
                                               ViReal64 range );

ViStatus _VI_FUNC AmetekDC_ConfigureCurrentLimit( ViSession vi,
                                                ViConstString channelName,
                                                ViInt32 behavior,
                                                ViReal64 limit );

ViStatus _VI_FUNC AmetekDC_ConfigureOVP( ViSession vi,
                                       ViConstString channelName,
                                       ViBoolean enabled,                                       
                                       ViReal64 limit );

ViStatus _VI_FUNC AmetekDC_ConfigureVoltageLevel( ViSession vi,
                                                ViConstString channelName,
                                                ViReal64 level );

ViStatus _VI_FUNC AmetekDC_GetChannelName( ViSession vi, ViInt32 index,
                                         ViInt32 bufferSize,
                                         ViChar name[] );


ViStatus _VI_FUNC AmetekDC_QueryOutputState( ViSession vi, 
                                           ViConstString channelName, 
                                           ViInt32 outputState, 
                                           ViBoolean *inState );                                                


ViStatus _VI_FUNC AmetekDC_QueryMaxCurrentLimit( ViSession vi,
                                               ViConstString channelName,
                                               ViReal64 VoltageLevel,
                                               ViReal64 *maxCurrentLimit );

ViStatus _VI_FUNC AmetekDC_QueryMaxVoltageLevel( ViSession vi,
                                               ViConstString channelName,
                                               ViReal64 CurrentLimit,
                                               ViReal64 *maxVoltageLevel );

ViStatus _VI_FUNC AmetekDC_ResetOutputProtection( ViSession vi,
                                                ViConstString channelName );

/*------- END: Ivi_DCPwr Required Utility Functions -------------------------------*/

/*------- START: Ivi_DCPwr Capability Groups Functions (optional) ------------------*/

	/*- IviDCPwr Measure Capability Functions -*/
ViStatus _VI_FUNC AmetekDC_Measure( ViSession vi, ViConstString channelName,
                                  ViInt32 measurementType, 
                                  ViReal64 *measurement );
	
/*------- END: Ivi_DCPwr Capability Groups Functions ------------------------------*/



/**************************************************************************** 
 *---------------- Instrument Specific Driver Function Declarations -----------------* 
 ****************************************************************************/


/*------- START: Multichannel Functions -----------------------------------------*/

	/*- Protection Mechanism Functions -*/                                                                                      
ViStatus _VI_FUNC AmetekDC_ConfigureProtectionSetpoint( ViSession vi,
                                                      ViConstString channelName,
                                                      ViAttr protectionType,
                                                      ViBoolean behavior, 
                                                      ViReal64 limit );


ViStatus _VI_FUNC AmetekDC_ConfigureProtectionLatch( ViSession vi,
                                                   ViConstString channelName,
                                                   ViAttr protectionType, 
                                                   ViBoolean state );
                                                  
ViStatus _VI_FUNC AmetekDC_ConfigureFoldbackSetting( ViSession vi, 
                                                   ViConstString channelName,
                                                   ViInt32 foldMode );

ViStatus _VI_FUNC AmetekDC_ConfigureFoldDelay( ViSession vi,
                                             ViConstString channelName, 
                                             ViReal64 value );

ViStatus _VI_FUNC AmetekDC_ConfigureFrontPanelTimeout( ViSession vi,
                                                ViConstString channelName,
                                                ViInt32 timeout);

ViStatus _VI_FUNC AmetekDC_SetRegister( ViSession vi,
                                      ViConstString channelName,
                                      ViInt32 setRegister, 
                                      ViInt16 setValue );

ViStatus _VI_FUNC AmetekDC_ClearStatus( ViSession vi, ViConstString channelName );

ViStatus _VI_FUNC AmetekDC_Preset( ViSession vi, ViConstString channelName );


                                                
/*------- General Query Functions -----------------------------------------*/

ViStatus _VI_FUNC AmetekDC_QuerySetpoint( ViSession vi, 
                                        ViConstString channelName,
                                        ViAttr AttrId, 
                                        ViReal64 *setpointLevel );


ViStatus _VI_FUNC AmetekDC_QueryRegister( ViSession vi,
                                        ViConstString channelName,
                                        ViInt32 queryRegister, 
                                        ViInt16 *registerValue );

ViStatus _VI_FUNC AmetekDC_QueryProtectionSettings( ViSession vi,
                                                  ViConstString channelName,
                                                  ViAttr protectionType,
                                                  ViBoolean *behavior, 
                                                  ViReal64 *level );

ViStatus _VI_FUNC AmetekDC_QueryProtectionLatch( ViSession vi,
                                               ViConstString channelName,
                                               ViAttr protectionType, 
                                               ViBoolean* state  );

ViStatus _VI_FUNC AmetekDC_QueryProtectionTripped( ViSession vi,
                                                 ViConstString channelName,
                                                 ViInt32 protectionType, 
                                                 ViBoolean *value );

ViStatus _VI_FUNC AmetekDC_QueryFrontPanelTimeout( ViSession vi,
                                                ViConstString channelName,
                                                ViInt32* timeout);

ViStatus _VI_FUNC AmetekDC_QueryEnabled( ViSession vi, ViConstString channelName,
                                       ViAttr attrId, ViBoolean *enabled );

ViStatus _VI_FUNC AmetekDC_QuerySource( ViSession vi, ViConstString channelName,
                                      ViAttr attrId, ViChar source[] );

ViStatus _VI_FUNC AmetekDC_QueryMode( ViSession vi, ViConstString channelName,
                                    ViAttr attrId, ViChar mode[] );



/*------- Auto Sequence Action Functions ----------------------------------*/
	/*- Recording/Programming Sequence Functions -*/
ViStatus _VI_FUNC AmetekDC_SetAutoSequenceRecordStart( 
                                                   ViSession vi,
                                                   ViConstString channelName );

ViStatus _VI_FUNC AmetekDC_SetAutoSequenceRecordStop( 
                                                   ViSession vi,
                                                   ViConstString channelName );

	/*- Deleting Sequence Functions -*/
ViStatus _VI_FUNC AmetekDC_DeleteAllSequences( ViSession vi,
                                             ViConstString channelName );

	/*- Programming Sequences Functions -*/
ViStatus _VI_FUNC AmetekDC_SetAutoSequenceRepeat( 
                                                   ViSession vi,
                                                   ViConstString channelName,
                                                   ViInt16 repeatCount );

	/*- Running Sequences Functions -*/ 

ViStatus _VI_FUNC AmetekDC_SendAutoSequenceSoftwareTrigger( 
                                                ViSession vi, 
                                                ViConstString channelName );

ViStatus _VI_FUNC AmetekDC_SetAutoSequenceState (ViSession instrumentHandle,
                                      ViConstString channelName,
                                      ViChar state);

ViStatus _VI_FUNC AmetekDC_SetAutoSequenceDwellTime( ViSession vi, 
					      ViConstString channelName,
					      ViInt32 delayTimeSec );

	/*- Query Auto Sequence Functions -*/
ViStatus _VI_FUNC AmetekDC_QueryAutoSequenceState( ViSession vi,
                                                     ViConstString channelName,
                                                     ViChar state[] );
 
ViStatus _VI_FUNC AmetekDC_QueryAutoSequenceRepeat( ViSession vi,
                                                      ViConstString channelName,
                                                      ViChar repeatCount[] );

ViStatus _VI_FUNC AmetekDC_QueryAutoSequenceDwellTime( ViSession vi, 
                                      ViConstString channelName,
                                      ViInt32* delayTime );


	/*- System Setup Functions -*/                                            
ViStatus _VI_FUNC AmetekDC_SetOutputEnabledPowerOn( ViSession vi,
                                                  ViConstString channelName,
                                                  ViBoolean enabled );

ViStatus _VI_FUNC AmetekDC_SetCurrentShareMode( ViSession vi,
                                              ViConstString channelName,
                                              ViConstString mode ); 

ViStatus _VI_FUNC AmetekDC_SaveUserSettings( ViSession vi,
                                           ViConstString channelName,
                                           ViInt16 userSetting );
                                          
ViStatus _VI_FUNC AmetekDC_RecallUserSetting( ViSession vi,
                                            ViConstString channelName,
                                            ViInt16 userSetting );


ViStatus _VI_FUNC AmetekDC_SetRemoteMode( ViSession vi,
                                        ViConstString channelName,
                                        ViInt32 state ); 

ViStatus _VI_FUNC AmetekDC_SelectVoltRemoteControlSource( ViSession vi,
                                                    ViConstString channelName,
                                                    ViInt32 source );

ViStatus _VI_FUNC AmetekDC_SelectCurrRemoteControlSource( ViSession vi,
                                                    ViConstString channelName,
                                                    ViInt32 source );
                                                   
ViStatus _VI_FUNC AmetekDC_SelectVoltAnalogProgrammingRange( ViSession vi,
                                                       ViConstString channelName,
                                                       ViReal64 range ); 

ViStatus _VI_FUNC AmetekDC_SelectCurrAnalogProgrammingRange( ViSession vi,
                                                       ViConstString channelName,
                                                       ViReal64 range ); 

	/*- Error Functions -*/
ViStatus _VI_FUNC AmetekDC_ResetMultichannelPowerSupply( 
                                             ViSession vi,
                                             ViConstString channelName );

ViStatus _VI_FUNC AmetekDC_ErrorQueryMultichannel( ViSession vi,
                                                 ViConstString channelName,
                                                 ViInt32 *errCode, 
                                                 ViChar errorMessage[] );

/*------- END: Multichannel Functions -----------------------------------------*/

	/*- Utility Functions the interact with the local system -*/
ViStatus _VI_FUNC AmetekDC_SetCommBaudRate( ViSession vi, ViInt32 uiDataRate );

ViStatus _VI_FUNC AmetekDC_GetCommBaudRate( ViSession vi, ViInt32* uiDataRate );

ViStatus _VI_FUNC AmetekDC_QueryAddress( ViSession vi, ViInt32 *address );

ViStatus _VI_FUNC AmetekDC_SetCommandTimeDelay( ViSession vi, ViReal64 delayTimeSec );

ViStatus _VI_FUNC AmetekDC_GetCommandTimeDelay( ViSession vi, ViReal64* delayTimeSec );

/*------- Instrument I/O Functions -------------------------------------------*/
ViStatus _VI_FUNC AmetekDC_WriteInstrData( ViSession vi, ViConstString writeBuffer );

ViStatus _VI_FUNC AmetekDC_ReadInstrData( ViSession vi, ViInt32 numBytes, 
                                        ViChar rdBuf[], ViInt32 *bytesRead );

ViStatus _VI_FUNC AmetekDC_SendCmd(ViSession vi, ViSession io,
                                 ViConstString channelName,ViUInt32 cmdIdx,
                                 ViConstString cmdParameters );

/*********************************************************
  Functions reserved for class driver use only.
  End-users should not call these functions.  
*********************************************************/
ViStatus _VI_FUNC  AmetekDC_IviInit( ViRsrc resourceName, ViBoolean IDQuery, 
                                   ViBoolean reset, ViSession vi );
                                   
ViStatus _VI_FUNC  AmetekDC_IviClose( ViSession vi );   


/*------ Deprecated Error Information functions ----------------------------*/
ViStatus _VI_FUNC AmetekDC_GetErrorInfo( ViSession vi, ViStatus *primaryError, 
                                       ViStatus *secondaryError, 
                                       ViChar errorElaboration[256] );
                                       
ViStatus _VI_FUNC AmetekDC_ClearErrorInfo( ViSession vi );


/**************************************************************************** 
 *---------------------------- End Include File ----------------------------* 
 ****************************************************************************/
#if defined(__cplusplus) || defined(__cplusplus__)
}
#endif
#endif /* __AmetekDC_HEADER */
